/******************************************************************************
 * Public header file for the Timer driver.
 * 
 * Freescale Semiconductor Inc.
 * (c) Copyright 2004-2012 Freescale Semiconductor, Inc.
 * ALL RIGHTS RESERVED.
 *
*******************************************************************************
 *
 * THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
***************************************************************************//*!
******************************************************************************/


#ifndef TIMER_H_
#define TIMER_H_




/******************************************************************************
*******************************************************************************
* Includes
*******************************************************************************
******************************************************************************/

#include "TMR_Interface.h"
#include "EmbeddedTypes.h"


/******************************************************************************
*******************************************************************************
* Private macros
*******************************************************************************
******************************************************************************/ 
/* SOFTWARE RELATED SECTION */
 
/******************************************************************************
 * NAME: NumberOfElements()
 * DESCRIPTION: calculates the number of elements in a given array
 *****************************************************************************/
#define NumberOfElements(array)   ((sizeof(array) / (sizeof(array[0]))))

/******************************************************************************
 * NAME: mTmrStatusFree_c
 * DESCRIPTION: self explanatory
 *****************************************************************************/
#define mTmrStatusFree_c    0x00

/******************************************************************************
 * NAME: mTmrStatusActive_c
 * DESCRIPTION: self explanatory
 *****************************************************************************/
#define mTmrStatusActive_c    0x20

/******************************************************************************
 * NAME: mTmrStatusReady_c
 * DESCRIPTION: self explanatory
 *****************************************************************************/
#define mTmrStatusReady_c    0x40

/******************************************************************************
 * NAME: mTmrStatusInactive_c
 * DESCRIPTION: self explanatory
 *****************************************************************************/
#define mTmrStatusInactive_c    0x80

/******************************************************************************
 * NAME: mTimerStatusMask_c
 * DESCRIPTION: timer status mask
 *****************************************************************************/
#define mTimerStatusMask_c      ( mTmrStatusActive_c \
| mTmrStatusReady_c \
| mTmrStatusInactive_c)

/******************************************************************************
 * NAME: TMR_IsTimerAllocated()
 * DESCRIPTION: checks if a specified timer is allocated
 *****************************************************************************/
#define TMR_IsTimerAllocated(timerID)   (maTmrTimerStatusTable[(timerID)])

/******************************************************************************
 * NAME: TMR_MarkTimerFree()
 * DESCRIPTION: marks the specified timer as free
 *****************************************************************************/
#define TMR_MarkTimerFree(timerID)       maTmrTimerStatusTable[(timerID)] = 0

/******************************************************************************
 * NAME: IsLowPowerTimer()
 * DESCRIPTION: Detect if the timer is a low-power timer
 *****************************************************************************/
#define IsLowPowerTimer(type)           ((type) & gTmrLowPowerTimer_c)

/******************************************************************************
 * NAME: mTimerType_c
 * DESCRIPTION: timer types
 *****************************************************************************/
#define mTimerType_c            ( gTmrSingleShotTimer_c \
	| gTmrSetSecondTimer_c \
	| gTmrSetMinuteTimer_c \
	| gTmrIntervalTimer_c )

/******************************************************************************
 * NAME: mTMR_Event_c
 * DESCRIPTION: TMR_Task() event flag. Only one event is needed.
 *****************************************************************************/
#define mTMR_Event_c    ( 1 << 0 )

/* HARDWARE RELATED SECTION */

/******************************************************************************
 * NAME: gTMR_TPMIrqNo
 * DESCRIPTION: TPMx interrupt number
 * VALID VALUES: refer to L2K manual
 *****************************************************************************/

#if(gTMR_TPMx == 0)
#define gTMR_TPMIrqNo                   (INT_TPM0-16)
#elif (gTMR_TPMx == 1)
#define gTMR_TPMIrqNo                   (INT_TPM1-16)
#else
#define gTMR_TPMIrqNo                   (INT_TPM2-16)
#endif

/******************************************************************************
 * NAME: gTMR_TPMInterruptPriority
 * DESCRIPTION: TPMx interrupt priority
 * VALID VALUES: 0..15
 *****************************************************************************/
#ifndef gTMR_TPMInterruptPriority
#define gTMR_TPMInterruptPriority       7
#endif


/******************************************************************************
 * NAME: gTmrTPM_max
 * DESCRIPTION: Maximum number of TPM hardware modules 
 * 			   available on target platform	 
 * VALID VALUES: 0-1 (refer to L2K manual)
 * 0 Stands for TPM0
 * 1 Standd for TPM1
 *****************************************************************************/

#ifndef gTmrTPM_max
#define gTmrTPM_max                     1 
#endif


/******************************************************************************
 * NAME: gTmrTPM_CN_max
 * DESCRIPTION: Maximum number of TPM channels
 * 			   available on target platform	  
 * VALID VALUES: 0-5 (refer to L2K manual)
 * 0 Stands for TPM Channel 0
 * 1 Standd for TPM Channel 1
 * 2 Stands for TPM Channel 2
 * 3 Stands for TPM Channel 3
 * 4 Standd for TPM Channel 4
 * 5 Stands for TPM Channel 5 
 *****************************************************************************/

#ifndef gTmrTPM_CN_max
#define gTmrTPM_CN_max                  5
#endif


/* Some checks ... */
#if ( gTMR_TPMx > gTmrTPM_max )
#error "TPM timer out of range [0-1]"
#endif

#if ( gTMR_TPM_CNx > gTmrTPM_CN_max )
#error "TPM channel out of range [0-5]"
#endif


/******************************************************************************
 * NAME: see below
 * DESCRIPTION: TPM registers definitions  
 * VALID VALUES: refer to L2K manual
 *****************************************************************************/


/*****************************************************************************/

#if(gTMR_TPMx == 0)
    #define gTPMxSC_c       		TPM0_SC                 /* TPM0 Status and Control register. */
    #define gTPMxCNT        		TPM0_CNT                /* TPM0 Counter register */
    #define gTPMxMOD_c     		TPM0_MOD                /* TPM0 Modulo register */


/******************************************************************************
 * NAME: see below
 * DESCRIPTION: TPM channel registers definition
 * VALID VALUES: refer to L2K manual
 *****************************************************************************/

    #if (gTMR_TPM_CNx == 0)
        #define gTPMxCnSC_c     	TPM0_C0SC               /* Channel 0 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C0V                /* Channel 0 Value register */
    #elif (gTMR_TPM_CNx == 1)
        #define gTPMxCnSC_c     	TPM0_C1SC               /* Channel 1 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C1V                /* Channel 1 Value register */
    #elif (gTMR_TPM_CNx == 2)
        #define gTPMxCnSC_c     	TPM0_C2SC               /* Channel 2 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C2V                /* Channel 2 Value register */
    #elif (gTMR_TPM_CNx == 3)
        #define gTPMxCnSC_c     	TPM0_C3SC               /* Channel 3 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C3V                /* Channel 3 Value register */
    #elif (gTMR_TPM_CNx == 4)
        #define gTPMxCnSC_c     	TPM0_C4SC               /* Channel 4 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C4V                /* Channel 4 Value register */
    #elif (gTMR_TPM_CNx == 5)
        #define gTPMxCnSC_c     	TPM0_C5SC               /* Channel 5 Status and Control register */
        #define gTPMxCnV_c      	TPM0_C5V                /* Channel 5 Value register */
    #endif
    
#elif (gTMR_TPMx == 1)
    #define gTPMxSC_c       		TPM1_SC                 /* TPM1 Status and Control register. */
    #define gTPMxCNT        		TPM1_CNT                /* TPM1 Counter register */
    #define gTPMxMOD_c     		TPM1_MOD                /* TPM1 Modulo register */


    /******************************************************************************
     * NAME: see below
     * DESCRIPTION: TPM channel registers definition
     * VALID VALUES: refer to L2K manual
     ******************************************************************************/
     
    #if (gTMR_TPM_CNx == 0)
        #define gTPMxCnSC_c     	TPM1_C0SC               /* Channel 0 Status and Control register */
        #define gTPMxCnV_c      	TPM1_C0V                /* Channel 0 Value register */
    #else
        #define gTPMxCnSC_c     	TPM1_C1SC               /* Channel 1 Status and Control register */
        #define gTPMxCnV_c      	TPM1_C1V                /* Channel 1 Value register */
    #endif

#else 
    #define gTPMxSC_c       		TPM2_SC                 /* TPM2 Status and Control register. */
    #define gTPMxCNT        		TPM2_CNT                /* TPM2 Counter register */
    #define gTPMxMOD_c     		TPM2_MOD                /* TPM2 Modulo register */

    
    /******************************************************************************
     * NAME: see below
     * DESCRIPTION: TPM channel registers definition
     * VALID VALUES: refer to L2K manual
     ******************************************************************************/
    #if (gTMR_TPM_CNx == 0)
        #define gTPMxCnSC_c     	TPM2_C0SC         		/* Channel 0 Status and Control register */
        #define gTPMxCnV_c      	TPM2_C0V          		/* Channel 0 Value register */
    #else
        #define gTPMxCnSC_c     	TPM2_C1SC         		/* Channel 1 Status and Control register */
        #define gTPMxCnV_c      	TPM2_C1V          		/* Channel 1 Value register */
    #endif
    
#endif

/******************************************************************************
 * NAME: gTmrSourceClkKHz_c
 * DESCRIPTION: Source timer clock in Khz
 * VALID VALUES: refer to L2k manual
 *****************************************************************************/
#define gTmrSourceClkKHz_c      24000

/*
 * NAME: gTPMxSC_PrescaleX_c
 * DESCRIPTION: Prescaler values
 * VALID VALUES: refer to Kinetis manual
 */
#define gTPMxSC_Prescale1_c       (0x00)    					/* For prescale Value of 1   */
#define gTPMxSC_Prescale2_c       (0x01)    					/* For prescale Value of 2   */
#define gTPMxSC_Prescale4_c       (0x02)    					/* For prescale Value of 4   */
#define gTPMxSC_Prescale8_c       (0x03)    					/* For prescale Value of 8   */
#define gTPMxSC_Prescale16_c      (0x04)    					/* For prescale Value of 16  */
#define gTPMxSC_Prescale32_c      (0x05)    					/* For prescale Value of 32  */
#define gTPMxSC_Prescale64_c      (0x06)    					/* For prescale Value of 64  */
#define gTPMxSC_Prescale128_c     (0x07)    					/* For prescale Value of 128 */

/*
 * NAME: gTPMx_TPMSRCx_c
 * DESCRIPTION: Clock source values
 * VALID VALUES: refer to L2k manual
 */
#define gTPMx_TPMSRCB_c                          0x2000000u             	/* SIM_SOPT2_TPMSRC, high bit. (RW) */
#define gTPMx_TPMSRCA_c                          0x1000000u             	/* SIM_SOPT2_TPMSRC, low bit. (RW) */

/*
The resolution of the timer is between 
4ms(max. time a task to be executed) - 0xffff * (1tick in ms).  
 */

/******************************************************************************
 * NAME: gTPMxSC_PrescaleCount_c
 * DESCRIPTION: Prescaler value used by TMR module
 *              Changing the prescaler the resolution will increase or decrease
 * VALID VALUES: refer to L2K manual
 *****************************************************************************/ 
//#define gTPMxSC_PrescaleCount_c     gTPMxSC_Prescale128_c
#define gTPMxSC_PrescaleCount_c     gTPMxSC_Prescale32_c

/******************************************************************************
 * NAME: see below
 * DESCRIPTION: Bits in the TPMx timer status and control registers (TPMxSC)
 * VALID VALUES: refer to L2K manual
 *****************************************************************************/
#define gTPMxSC_TOF_c           TPM_SC_TOF_MASK    			        /* Timer overflow flag. (RO) */
#define gTPMxSC_TOIE_c          TPM_SC_TOIE_MASK    		                /* Timer overflow interrupt enable. (RW) */
#define gTPMxSC_CPWMS_c         TPM_SC_CPWMS_MASK    			        /* Center-aliged PWM select. (RW) */
#define gTPMxSC_CMODB_c         TPM_SC_CMOD(0x02)   				/* Clock Mode Selection, high bit. (RW) */
#define gTPMxSC_CMODA_c         TPM_SC_CMOD(0x01)    				/* Clock Mode Selection, low bit. (RW) */
#define gTPMxSC_PS2_c           TPM_SC_PS(0x04)   				/* Prescale divisor select, high bit. (RW) */
#define gTPMxSC_PS1_c           TPM_SC_PS(0x02) 				/* Prescale divisor select, middle bit. (RW) */
#define gTPMxSC_PS0_c           TPM_SC_PS(0x01) 				/* Prescale divisor select, low bit. (RW) */

/******************************************************************************
 * NAME: see below
 * DESCRIPTION: Bits in the TPMx channel N status and control registers (TPMxCnSC)
 * VALID VALUES: refer to Kinetis manual
 *****************************************************************************/
#define gTPMxCnSC_CHF_c	        TPM_CnSC_CHF_MASK    					/* Channel Overflow flag. */
#define gTPMxCnSC_CHIE_c        TPM_CnSC_CHIE_MASK    					/* Channel Interrupt enable. */        
#define gTPMxCnSC_MSB_c         TPM_CnSC_MSB_MASK    					/* Channel Mode select bit B. */
#define gTPMxCnSC_MSA_c         TPM_CnSC_MSA_MASK    					/* Channel Mode select bit A. */
#define gTPMxCnSC_ELSB_c        TPM_CnSC_ELSB_MASK    					/* Channel Edge/level select high bit. */
#define gTPMxCnSC_ELSA_c        TPM_CnSC_ELSA_MASK    					/* Channel Edge/level select low bit. */
#define gTPMxCnSC_DMA_c         TPM_CnSC_DMA_MASK    					/* DMA transfer enable bit */

/******************************************************************************
 * NAME: gTPMxSC_ClockSource_c
 * DESCRIPTION: Clock source for TMR module
 * VALID VALUES: refer to Kinetis manual
 *****************************************************************************/
#define gTPMxSC_ClockSource_c	gTPMx_TPMSRCA_c			                /* Select the system clock. */  

/******************************************************************************
 * NAME: gTPMxSC_Stop_c
 * DESCRIPTION: TPMx_SC register value for disabling (stopping) the timer
 * VALID VALUES: refer to Kinetis manual
 *****************************************************************************/
#define gTPMxSC_Stop_c          0x00

/******************************************************************************
 * NAME: gTPMxSC_ClearRegister_c
 * DESCRIPTION: Macro used to clear the TPMx_SC register
 *****************************************************************************/
#define gTPMxSC_ClearRegister_c 0x00

/******************************************************************************
 * NAME: TPMReadCnVRegister()
 * DESCRIPTION: Macro used to read the compare registers 
 *****************************************************************************/
#define TPMReadCnVRegister(variable)	(variable) = gTPMxCnV_c;

/******************************************************************************
 * NAME: gTPMxCnSC_c_ClearRegister_c
 * DESCRIPTION: Macro used to clear the gTPMxCnSC_c register
 *****************************************************************************/
#define gTPMxCnSC_c_ClearRegister_c 0x00

/******************************************************************************
 * NAME: TPMReadCNTRegister()
 * DESCRIPTION: Macro used to read the free running counter
 *****************************************************************************/
#define TPMReadCNTRegister(variable)   	(variable) = gTPMxCNT;

/******************************************************************************
 * NAME: gTPMxCNT_ClearRegister_c
 * DESCRIPTION: Macro used to clear the gTPMxCNT register
 *****************************************************************************/
#define gTPMxCNT_ClearRegister_c 0x00

/******************************************************************************
 * NAME: TPMReadCNTRegister()
 * DESCRIPTION: Enable the hardware timer
 *****************************************************************************/
#define TPMStartTimerHardware()		(gTPMxSC_c |= (gTPMxSC_CMODA_c | gTPMxSC_PrescaleCount_c))       

/******************************************************************************
 * NAME: TPMReadCNTRegister()
 * DESCRIPTION: Disable the hardware timer
 *****************************************************************************/
#define TPMStopTimerHardware()		(gTPMxSC_c = gTPMxSC_Stop_c)

/******************************************************************************
*******************************************************************************
* Private type definitions
*******************************************************************************
******************************************************************************/

/******************************************************************************
 * Type name: tmrTimerStatus_t
 * Type description: The status and type are bitfields, to save RAM.
 *                   This costs some code space, though.
 * Members: N/A
 *****************************************************************************/
typedef uint8_t tmrTimerStatus_t;

/******************************************************************************
 * NAME: tmrStatus_t
 * DESCRIPTION: timer status - see the status macros.
 *              If none of these flags are on, the timer is not allocated.
 *              For allocated timers, exactly one of these flags will be set.
 *              mTmrStatusActive_c - Timer has been started and has not yet expired.
 *              mTmrStatusReady_c - TMR_StartTimer() has been called for this timer, but
 *                                  the timer task has not yet actually started it. The
 *                                  timer is considered to be active.
 *              mTmrStatusInactive_c Timer is allocated, but is not active.
 *****************************************************************************/
typedef uint8_t tmrStatus_t;

/******************************************************************************
 * NAME: tmrTimerTicks_t
 * DESCRIPTION: 16-bit timer ticks type definition
 * VALID VALUES: see definition
 *****************************************************************************/
typedef uint16_t tmrTimerTicks16_t;

/******************************************************************************
 * NAME: tmrTimerTicks_t
 * DESCRIPTION: 32-bit timer ticks type definition
 * VALID VALUES: see definition
 *****************************************************************************/
typedef uint32_t tmrTimerTicks32_t;

/******************************************************************************
 * Type name: tmrTimerTableEntry_tag
 * Type description: One entry in the main timer table.
 * Members: intervalInTicks - The timer's original duration, in ticks.
 *                            Used to reset intervnal timers.
 *
 *          countDown - When a timer is started, this is set to the duration.
 *                      The timer task decrements this value. When it reaches
 *                      zero, the timer has expired.
 *          pfCallBack - Pointer to the callback function
 *****************************************************************************/
typedef struct tmrTimerTableEntry_tag {
  tmrTimerTicks32_t intervalInTicks;
  tmrTimerTicks32_t remainingTicks;
  pTPMrCallBack_t pfCallBack;
} tmrTimerTableEntry_t;

#endif /* TIMER_H_ */
